/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ui.internal.commands;

import java.text.Collator;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

public final class Configuration implements Comparable {

	private final static int HASH_FACTOR = 57;
	private final static int HASH_INITIAL = 67;

	private static Comparator nameComparator;
	
	public static Configuration create(String description, String id, String name, String parent, String plugin)
		throws IllegalArgumentException {
		return new Configuration(description, id, name, parent, plugin);
	}

	public static Comparator nameComparator() {
		if (nameComparator == null)
			nameComparator = new Comparator() {
				public int compare(Object left, Object right) {
					return Collator.getInstance().compare(((Configuration) left).name, ((Configuration) right).name);
				}	
			};		
		
		return nameComparator;
	}

	public static SortedMap sortedMapById(List configurations)
		throws IllegalArgumentException {
		if (configurations == null)
			throw new IllegalArgumentException();

		SortedMap sortedMap = new TreeMap();			
		Iterator iterator = configurations.iterator();
		
		while (iterator.hasNext()) {
			Object object = iterator.next();
			
			if (!(object instanceof Configuration))
				throw new IllegalArgumentException();
				
			Configuration configuration = (Configuration) object;
			sortedMap.put(configuration.id, configuration);									
		}			
		
		return sortedMap;
	}

	public static SortedMap sortedMapByName(List configurations)
		throws IllegalArgumentException {
		if (configurations == null)
			throw new IllegalArgumentException();

		SortedMap sortedMap = new TreeMap();			
		Iterator iterator = configurations.iterator();
		
		while (iterator.hasNext()) {
			Object object = iterator.next();
			
			if (!(object instanceof Configuration))
				throw new IllegalArgumentException();
				
			Configuration configuration = (Configuration) object;
			sortedMap.put(configuration.name, configuration);									
		}			
		
		return sortedMap;
	}

	private String description;
	private String id;
	private String name;
	private String parent;
	private String plugin;
	
	private Configuration(String description, String id, String name, String parent, String plugin)
		throws IllegalArgumentException {
		super();
		
		if (id == null || name == null)
			throw new IllegalArgumentException();
		
		this.description = description;
		this.id = id;
		this.name = name;
		this.parent = parent;
		this.plugin = plugin;
	}
	
	public int compareTo(Object object) {
		Configuration configuration = (Configuration) object;
		int compareTo = id.compareTo(configuration.id);
		
		if (compareTo == 0) {		
			compareTo = name.compareTo(configuration.name);			
		
			if (compareTo == 0) {
				Util.compare(description, configuration.description);
				
				if (compareTo == 0) {
					compareTo = Util.compare(parent, configuration.parent);

					if (compareTo == 0)
						compareTo = Util.compare(plugin, configuration.plugin);								
				}							
			}
		}
		
		return compareTo;	
	}
	
	public boolean equals(Object object) {
		if (!(object instanceof Configuration))
			return false;

		Configuration configuration = (Configuration) object;	
		return Util.equals(description, configuration.description) && id.equals(configuration.id) && name.equals(configuration.name) && Util.equals(parent, configuration.parent) && 
			Util.equals(plugin, configuration.plugin);
	}

	public String getDescription() {
		return description;	
	}
	
	public String getId() {
		return id;	
	}
	
	public String getName() {
		return name;
	}	

	public String getParent() {
		return parent;
	}

	public String getPlugin() {
		return plugin;
	}

	public int hashCode() {
		int result = HASH_INITIAL;
		result = result * HASH_FACTOR + Util.hashCode(description);
		result = result * HASH_FACTOR + id.hashCode();
		result = result * HASH_FACTOR + name.hashCode();
		result = result * HASH_FACTOR + Util.hashCode(parent);
		result = result * HASH_FACTOR + Util.hashCode(plugin);
		return result;
	}
	
	public String toString() {
		return name + " (" + id + ')';	 //$NON-NLS-1$
	}
}
